import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.HBaseConfiguration;
import org.apache.hadoop.hbase.client.*;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.hbase.filter.*;
import org.apache.hadoop.hbase.filter.CompareFilter.CompareOp;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

public class HbaseClientExample {

	private HTableInterface twitterTable;

	public HbaseClientExample() {
		try {
			Configuration conf = HBaseConfiguration.create();
			twitterTable = new HTable(conf, "twitter");
		} catch (IOException e) {
			e.printStackTrace();
		}
	}

	public void close() {
		try {
			twitterTable.close();
		} catch (IOException e) {
			e.printStackTrace();
		}
	}
	
	public void updateTwitter(Twitter twitter) {
		this.insertTwitter(twitter);
	}

	public void insertTwitter(Twitter twitter) {
		
		
		
		
		
		
		// Create a new Put object with the Row Key as the bytes of the twitter id
		Put put = new Put(Bytes.toBytes(twitter.getTwitterId()));
		
		// Add "text" and "createdAt" to the "general" column family
		if (twitter.getText() != null) {
			put.add(Bytes.toBytes("general"), Bytes.toBytes("text"), Bytes.toBytes(twitter.getText()));
		}		
		if (twitter.getCreatedAt() != null) {
			put.add(Bytes.toBytes("general"), Bytes.toBytes("createdAt"), Bytes.toBytes(twitter.getCreatedAt()));
		}

		// Add the "name" to the "user" column family
		if (twitter.getName() != null) {
			put.add(Bytes.toBytes("user"), Bytes.toBytes("name"), Bytes.toBytes(twitter.getName()));
		}
		
		try {
			// Add this twitter to the "twitter" table
			twitterTable.put(put);
		} catch (IOException e) {
			e.printStackTrace();
		}
	}

	public Twitter getTargetTwitter(String rowkey) {
		try {

			// Create a Get object with the rowkey (as a byte[])
			Get get = new Get(Bytes.toBytes(rowkey));
			
			// only retrieve the *user* column family
			get.addFamily(Bytes.toBytes("user"));

			// Execute the Get
			Result result = twitterTable.get(get);

			// Retrieve the results
			Twitter twitter = new Twitter();
			byte[] rowKeyBytes = result.getRow();	//get the row key 
			twitter.setTwitterId(Bytes.toString(rowKeyBytes));
			
			//get the cell value for columns in *user* column family
			byte[] valueBytes = result.getValue(Bytes.toBytes("user"), Bytes.toBytes("name"));
			twitter.setName(Bytes.toString(valueBytes));

			// Return the newly constructed Twitter
			return twitter;
		} catch (IOException e) {
			e.printStackTrace();
		}
		return null;
	}

	public void delete(String rowkey) {
		try {
			Delete delete = new Delete(Bytes.toBytes(rowkey));
			twitterTable.delete(delete);
		} catch (IOException e) {
			e.printStackTrace();
		}
	}

	public List<Twitter> filter(String pattern) {
		try {
			// Build a list to hold our results
			List<Twitter> twitterResults = new ArrayList<Twitter>();

			// Create and execute a scan
			Scan scan = new Scan();
			scan.addFamily(Bytes.toBytes("general"));	//get all columns from the specified family "general"
                        
                        scan.setFilter(new ValueFilter(CompareOp.EQUAL, new SubstringComparator("iphone6")));

			ResultScanner results = twitterTable.getScanner(scan);

			for (Result result : results) {
				
				// Build a new Twitter
				Twitter twitter = new Twitter();
				byte[] rowKeyBytes = result.getRow();	//get the row key 
				twitter.setTwitterId(Bytes.toString(rowKeyBytes));
				
				//get the cell value for columns in *general* column family
				byte[] valueBytes = result.getValue(Bytes.toBytes("general"), Bytes.toBytes("text"));
				twitter.setText(Bytes.toString(valueBytes));
				valueBytes = result.getValue(Bytes.toBytes("general"), Bytes.toBytes("createdAt"));
				twitter.setCreatedAt(Bytes.toString(valueBytes));
				
				// Add the Twitter to our results
				twitterResults.add(twitter);
			}

			// Return our results
			return twitterResults;
		} catch (IOException e) {
			e.printStackTrace();
		}
                return null;
        }

	public List<Twitter> fullScan() {
		try {
			// Build a list to hold our results
			List<Twitter> twitterResults = new ArrayList<Twitter>();

			// Create and execute a scan
			Scan scan = new Scan();
			scan.addFamily(Bytes.toBytes("general"));	//get all columns from the specified family "general"
			ResultScanner results = twitterTable.getScanner(scan);

			for (Result result : results) {
				
				// Build a new Twitter
				Twitter twitter = new Twitter();
				byte[] rowKeyBytes = result.getRow();	//get the row key 
				twitter.setTwitterId(Bytes.toString(rowKeyBytes));
				
				//get the cell value for columns in *general* column family
				byte[] valueBytes = result.getValue(Bytes.toBytes("general"), Bytes.toBytes("text"));
				twitter.setText(Bytes.toString(valueBytes));
				valueBytes = result.getValue(Bytes.toBytes("general"), Bytes.toBytes("createdAt"));
				twitter.setCreatedAt(Bytes.toString(valueBytes));
				
				// Add the Twitter to our results
				twitterResults.add(twitter);
			}

			// Return our results
			return twitterResults;
		} catch (IOException e) {
			e.printStackTrace();
		}
                return null;
        }

	public List<Twitter> rangeScan(String startRowKey, String endRowKey) {
		try {
			// Build a list to hold our results
			List<Twitter> twitterResults = new ArrayList<Twitter>();

			// Create and execute a scan
			Scan scan = new Scan(Bytes.toBytes(startRowKey), Bytes.toBytes(endRowKey));
			scan.addFamily(Bytes.toBytes("general"));	//get all columns from the specified family "general"
			ResultScanner results = twitterTable.getScanner(scan);

			for (Result result : results) {
				
				// Build a new Twitter
				Twitter twitter = new Twitter();
				byte[] rowKeyBytes = result.getRow();	//get the row key 
				twitter.setTwitterId(Bytes.toString(rowKeyBytes));
				
				//get the cell value for columns in *general* column family
				byte[] valueBytes = result.getValue(Bytes.toBytes("general"), Bytes.toBytes("text"));
				twitter.setText(Bytes.toString(valueBytes));
				valueBytes = result.getValue(Bytes.toBytes("general"), Bytes.toBytes("createdAt"));
				twitter.setCreatedAt(Bytes.toString(valueBytes));
				
				// Add the Twitter to our results
				twitterResults.add(twitter);
			}

			// Return our results
			return twitterResults;
		} catch (IOException e) {
			e.printStackTrace();
		}
		return null;
	}

	public static void main(String[] args) {
		
		HbaseClientExample example = new HbaseClientExample();

		// **************************************Create Operation*******************************************
                // Create five twitters; note that hbase table is *sparse*, i.e., not every column's value is specified.
		example.insertTwitter(new Twitter("twitter1", "iphone6 is awesome!", 
				"Fri Dec 12 13:36:55 +0000 2014", "John"));
		example.insertTwitter(new Twitter("twitter2", "Is iphone6 plus too big for one-hand use?", 
				null, "Bob"));
		example.insertTwitter(new Twitter("twitter3", "Merry Christmas, let's have fun", 
				"Wed Dec 25 01:13:17 +0000 2014", null));
		example.insertTwitter(new Twitter("twitter4", "Third straight win in the 2015 Asian Cup? What on EARTH has happened to the Chinese football team", 
				"Mon Jan 19 09:52:14 +0000 2015", "Alice"));
		example.insertTwitter(new Twitter("twitter5", "slow mo camera is best feature on iphone6", 
				null, null));

                // **************************************Retrieve Operation*****************************************
		// Execute a scan for row key range from "twitter2" to "twitter4"; two parameters [startRowKey, endRowKey)
		List<Twitter> scannedTwitters = example.rangeScan("twitter2", "twitter5");
		if (scannedTwitters != null) {
			System.out.println("Twitters within range [twitter2, twitter5):");
			for (Twitter twitter : scannedTwitters) {
				System.out.println(twitter.outputGeneralCF());
			}
		}

		// Get a specific row
		Twitter targetTwitter = example.getTargetTwitter("twitter1");
                System.out.println("Twitter with id *twitter1*:");
		System.out.println(targetTwitter.outputUserCF());

                // **************************************Update Operation*******************************************
                // update a cell for a specific row key, column family, and column qualifier
		example.updateTwitter(new Twitter("twitter2", "I can handle iphone6 plus with one hand now.", 
				"", "Bob"));
                
                // **************************************Delete Operation******************************************
		// Delete a row
		example.delete("twitter5");

		// Execute a complete table scan to see the effectiveness of deleting a row
		scannedTwitters.clear();
		scannedTwitters = example.fullScan();
		if (scannedTwitters != null) {
			System.out.println("All twitters after update and delete:");
			for (Twitter twitter : scannedTwitters) {
				System.out.println(twitter.outputGeneralCF());
			}
		}

                // *************************************Advanced: Filter******************************************
                // get the rows whose cell value contains *iphone6*
                scannedTwitters.clear();
		scannedTwitters = example.filter("iphone6");
		if (scannedTwitters != null) {
			System.out.println("Twitters mentioning iphone6");
			for (Twitter twitter : scannedTwitters) {
				System.out.println(twitter.outputGeneralCF());
			}
		}
                

		// Close our table
		example.close();
	}
}

class Twitter {

	private String twitterId;
	private String text;
	private String createdAt;
	private String name;

	public Twitter() {
	}

	public Twitter(String twitterId, String text, String createdAt, String name) {
		this.twitterId = twitterId;
		this.text = text;
		this.createdAt = createdAt;
		this.name = name;
	}

	public String getTwitterId() {
		return twitterId;
	}

	public void setTwitterId(String twitterId) {
		this.twitterId = twitterId;
	}

	public String getText() {
		return text;
	}

	public void setText(String text) {
		this.text = text;
	}

	public String getCreatedAt() {
		return createdAt;
	}

	public void setCreatedAt(String createdAt) {
		this.createdAt = createdAt;
	}

	public String getName() {
		return name;
	}

	public void setName(String name) {
		this.name = name;
	}
	
	public String outputGeneralCF() {
		return "\tTwitter ID: " + this.getTwitterId() + "\n\t\tgeneral:text: " + this.getText() 
				+ "\n\t\tgeneral:createdAt: " + this.getCreatedAt();
	}
	
	public String outputUserCF() {
		return "\tTwitter ID: " + this.getTwitterId() + "\n\t\tuser:name: " + this.getName();
	}
}
